import { useState, useRef, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Apple, Target, TrendingUp, TrendingDown, Heart, AlertCircle, Calendar, CheckCircle, Award, Star, Users, Zap, Shield } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedGoal, setSelectedGoal] = useState('')
  const [consultationDate, setConsultationDate] = useState('')
  const [showSuccess, setShowSuccess] = useState(false)

  const goals = [
    {
      id: 'weight-loss',
      name: 'Похудение',
      description: 'Снижение веса с сохранением мышечной массы',
      icon: TrendingDown,
      color: 'var(--nutrition-green)'
    },
    {
      id: 'weight-gain',
      name: 'Набор массы',
      description: 'Увеличение веса за счет мышечной массы',
      icon: TrendingUp,
      color: 'var(--nutrition-orange)'
    },
    {
      id: 'healthy-lifestyle',
      name: 'ЗОЖ',
      description: 'Поддержание здорового образа жизни',
      icon: Heart,
      color: 'var(--nutrition-teal)'
    },
    {
      id: 'allergies',
      name: 'Аллергии',
      description: 'Питание при пищевых аллергиях и непереносимостях',
      icon: AlertCircle,
      color: 'var(--nutrition-coral)'
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedGoal || !consultationDate) {
      alert('Пожалуйста, выберите цель и дату консультации')
      return
    }

    const goalData = goals.find(g => g.id === selectedGoal)
    
    const newConsultation = {
      id: Date.now(),
      plan: goalData.name,
      date: consultationDate,
      rating: null
    }

    const existing = JSON.parse(localStorage.getItem('nutriConsultations') || '[]')
    existing.push(newConsultation)
    localStorage.setItem('nutriConsultations', JSON.stringify(existing))

    setShowSuccess(true)
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  return (
    <div className="enroll-page-nutrition">
      <div className="container">
        <section className="enroll-header-nutrition">
          <div className="header-badge-nutrition">
            <Apple size={20} />
            <span>ЗАПИСЬ НА КОНСУЛЬТАЦИЮ</span>
          </div>
          <h1 className="enroll-title-nutrition">
            Записаться на <span className="title-accent-nutrition">консультацию</span>
          </h1>
          <p className="enroll-description-nutrition">
            Выберите вашу цель и дату консультации с профессиональным нутрициологом
          </p>
        </section>

        <section className="enroll-form-section-nutrition">
          <form className="form-nutrition" onSubmit={handleSubmit}>
            <div className="goals-header-nutrition">
              <h2 className="goals-title-nutrition">Выберите вашу цель</h2>
              <p className="goals-description-nutrition">
                Каждая цель требует индивидуального подхода к питанию
              </p>
            </div>
            <div className="goals-selection-nutrition">
              {goals.map((goal) => {
                const Icon = goal.icon
                return (
                  <button
                    key={goal.id}
                    type="button"
                    className={`goal-card-nutrition ${selectedGoal === goal.id ? 'selected' : ''}`}
                    onClick={() => setSelectedGoal(goal.id)}
                    style={{ '--goal-color': goal.color }}
                  >
                    <div className="goal-icon-wrapper-nutrition">
                      <Icon size={40} />
                    </div>
                    <h3 className="goal-name-nutrition">{goal.name}</h3>
                    <p className="goal-desc-nutrition">{goal.description}</p>
                    {selectedGoal === goal.id && (
                      <div className="check-badge-nutrition">
                        <CheckCircle size={24} />
                      </div>
                    )}
                  </button>
                )
              })}
            </div>

            {selectedGoal && (
              <div className="date-section-nutrition">
                <div className="date-label-nutrition">
                  <Calendar size={20} />
                  <span>Дата консультации</span>
                </div>
                <input
                  type="date"
                  className="date-input-nutrition"
                  value={consultationDate}
                  onChange={(e) => setConsultationDate(e.target.value)}
                  min={new Date().toISOString().split('T')[0]}
                  required
                />
              </div>
            )}

            <button 
              type="submit" 
              className="submit-btn-nutrition"
              disabled={!selectedGoal || !consultationDate}
            >
              <CheckCircle size={20} />
              <span>Записаться на консультацию</span>
            </button>

            {showSuccess && (
              <div className="success-message-nutrition">
                <Award size={24} />
                <span>Запись успешно оформлена!</span>
              </div>
            )}
          </form>
        </section>

        <ImageSlider />

        <BenefitsSection />
      </div>
    </div>
  )
}

const BenefitsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const benefits = [
    {
      icon: Shield,
      title: 'Индивидуальный подход',
      description: 'План питания разрабатывается специально для вас с учетом всех особенностей',
      color: 'var(--nutrition-green)'
    },
    {
      icon: Users,
      title: 'Опытные специалисты',
      description: 'Работаем только с сертифицированными нутрициологами с многолетним опытом',
      color: 'var(--nutrition-teal)'
    },
    {
      icon: Zap,
      title: 'Быстрые результаты',
      description: 'Первые положительные изменения вы заметите уже через 2-3 недели',
      color: 'var(--nutrition-orange)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`benefits-section-nutrition ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="benefits-header-nutrition">
          <h2 className="benefits-title-nutrition">Почему выбирают нас</h2>
          <p className="benefits-description-nutrition">
            Преимущества работы с нашими нутрициологами
          </p>
        </div>
        <div className="benefits-grid-nutrition">
          {benefits.map((benefit, index) => {
            const Icon = benefit.icon
            return (
              <div 
                key={index} 
                className="benefit-card-nutrition"
                style={{ 
                  animationDelay: `${index * 0.15}s`,
                  '--benefit-color': benefit.color
                }}
              >
                <div className="benefit-icon-wrapper-nutrition">
                  <Icon size={40} />
                </div>
                <h3 className="benefit-title-nutrition">{benefit.title}</h3>
                <p className="benefit-text-nutrition">{benefit.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

export default Enroll

